<?php
require_once 'config/database.php';

/**
 * Product Model
 * Handles all product-related database operations
 */
class ProductModel {
    private $conn;
    
    public function __construct() {
        global $conn;
        $this->conn = $conn;
    }
    
    /**
     * Get all products
     * @return array
     */
    public function getAllProducts() {
        $sql = "SELECT * FROM products ORDER BY created_at DESC";
        $result = $this->conn->query($sql);
        
        $products = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $products[] = $row;
            }
        }
        
        return $products;
    }
    
    /**
     * Get product by ID
     * @param int $id
     * @return array|null
     */
    public function getProductById($id) {
        $sql = "SELECT * FROM products WHERE id = ?";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            return $result->fetch_assoc();
        }
        
        return null;
    }
    
    /**
     * Get products by category
     * @param string $category
     * @return array
     */
    public function getProductsByCategory($category) {
        $sql = "SELECT * FROM products WHERE category = ? ORDER BY created_at DESC";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("s", $category);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $products = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $products[] = $row;
            }
        }
        
        return $products;
    }
    
    /**
     * Get best seller products
     * @return array
     */
    public function getBestSellers() {
        $sql = "SELECT * FROM products WHERE is_best_seller = 1 ORDER BY created_at DESC";
        $result = $this->conn->query($sql);
        
        $products = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $products[] = $row;
            }
        }
        
        return $products;
    }
    
    /**
     * Create new product
     * @param array $data
     * @return bool
     */
    public function createProduct($data) {
        $sql = "INSERT INTO products (name, category, description, image_url, price, is_best_seller, icon) 
                VALUES (?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("ssssdis", 
            $data['name'], 
            $data['category'], 
            $data['description'], 
            $data['image_url'], 
            $data['price'], 
            $data['is_best_seller'],
            $data['icon']
        );
        
        return $stmt->execute();
    }
    
    /**
     * Update product
     * @param int $id
     * @param array $data
     * @return bool
     */
    public function updateProduct($id, $data) {
        $sql = "UPDATE products SET name = ?, category = ?, description = ?, image_url = ?, 
                price = ?, is_best_seller = ?, icon = ?, updated_at = CURRENT_TIMESTAMP 
                WHERE id = ?";
        
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("ssssdisi", 
            $data['name'], 
            $data['category'], 
            $data['description'], 
            $data['image_url'], 
            $data['price'], 
            $data['is_best_seller'],
            $data['icon'],
            $id
        );
        
        return $stmt->execute();
    }
    
    /**
     * Delete product
     * @param int $id
     * @return bool
     */
    public function deleteProduct($id) {
        $sql = "DELETE FROM products WHERE id = ?";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("i", $id);
        
        return $stmt->execute();
    }
    
    /**
     * Get product categories
     * @return array
     */
    public function getCategories() {
        $sql = "SELECT DISTINCT category FROM products ORDER BY category";
        $result = $this->conn->query($sql);
        
        $categories = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $categories[] = $row['category'];
            }
        }
        
        return $categories;
    }
    
    /**
     * Count total products
     * @return int
     */
    public function countProducts() {
        $sql = "SELECT COUNT(*) as total FROM products";
        $result = $this->conn->query($sql);
        $row = $result->fetch_assoc();
        
        return $row['total'];
    }
}
?>