<?php
require_once 'config/database.php';

/**
 * Agenda Model
 * Handles all agenda-related database operations
 */
class AgendaModel {
    private $conn;
    
    public function __construct() {
        global $conn;
        $this->conn = $conn;
    }
    
    /**
     * Get all agenda items
     * @return array
     */
    public function getAllAgenda() {
        $sql = "SELECT * FROM agenda ORDER BY date ASC";
        $result = $this->conn->query($sql);
        
        $agenda = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $agenda[] = $row;
            }
        }
        
        return $agenda;
    }
    
    /**
     * Get agenda by ID
     * @param int $id
     * @return array|null
     */
    public function getAgendaById($id) {
        $sql = "SELECT * FROM agenda WHERE id = ?";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            return $result->fetch_assoc();
        }
        
        return null;
    }
    
    /**
     * Get current month agenda
     * @return array
     */
    public function getCurrentMonthAgenda() {
        $currentMonth = date('n');
        $currentYear = date('Y');
        
        $sql = "SELECT * FROM agenda WHERE month = ? AND year = ? ORDER BY date ASC";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("ii", $currentMonth, $currentYear);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $agenda = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $agenda[] = $row;
            }
        }
        
        return $agenda;
    }
    
    /**
     * Get agenda by month and year
     * @param int $month
     * @param int $year
     * @return array
     */
    public function getAgendaByMonth($month, $year) {
        $sql = "SELECT * FROM agenda WHERE month = ? AND year = ? ORDER BY date ASC";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("ii", $month, $year);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $agenda = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $agenda[] = $row;
            }
        }
        
        return $agenda;
    }
    
    /**
     * Get upcoming agenda (from today onwards)
     * @return array
     */
    public function getUpcomingAgenda() {
        $today = date('Y-m-d');
        
        $sql = "SELECT * FROM agenda WHERE date >= ? ORDER BY date ASC LIMIT 5";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("s", $today);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $agenda = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $agenda[] = $row;
            }
        }
        
        return $agenda;
    }
    
    /**
     * Create new agenda item
     * @param array $data
     * @return bool
     */
    public function createAgenda($data) {
        $sql = "INSERT INTO agenda (date, title, description, month, year) 
                VALUES (?, ?, ?, ?, ?)";
        
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("sssii", 
            $data['date'], 
            $data['title'], 
            $data['description'], 
            $data['month'], 
            $data['year']
        );
        
        return $stmt->execute();
    }
    
    /**
     * Update agenda item
     * @param int $id
     * @param array $data
     * @return bool
     */
    public function updateAgenda($id, $data) {
        $sql = "UPDATE agenda SET date = ?, title = ?, description = ?, 
                month = ?, year = ?, updated_at = CURRENT_TIMESTAMP 
                WHERE id = ?";
        
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("sssiii", 
            $data['date'], 
            $data['title'], 
            $data['description'], 
            $data['month'], 
            $data['year'],
            $id
        );
        
        return $stmt->execute();
    }
    
    /**
     * Delete agenda item
     * @param int $id
     * @return bool
     */
    public function deleteAgenda($id) {
        $sql = "DELETE FROM agenda WHERE id = ?";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("i", $id);
        
        return $stmt->execute();
    }
    
    /**
     * Get agenda for specific date range
     * @param string $startDate
     * @param string $endDate
     * @return array
     */
    public function getAgendaByDateRange($startDate, $endDate) {
        $sql = "SELECT * FROM agenda WHERE date BETWEEN ? AND ? ORDER BY date ASC";
        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("ss", $startDate, $endDate);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $agenda = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $agenda[] = $row;
            }
        }
        
        return $agenda;
    }
    
    /**
     * Count total agenda items
     * @return int
     */
    public function countAgenda() {
        $sql = "SELECT COUNT(*) as total FROM agenda";
        $result = $this->conn->query($sql);
        $row = $result->fetch_assoc();
        
        return $row['total'];
    }
}
?>