<?php
require_once 'models/TestimonialModel.php';

/**
 * Testimonial Controller
 * Handles all testimonial-related operations
 */
class TestimonialController {
    private $testimonialModel;
    
    public function __construct() {
        $this->testimonialModel = new TestimonialModel();
    }
    
    /**
     * Get all testimonials
     * @return array
     */
    public function getAllTestimonials() {
        return $this->testimonialModel->getAllTestimonials();
    }
    
    /**
     * Get testimonial by ID
     * @param int $id
     * @return array|null
     */
    public function getTestimonialById($id) {
        return $this->testimonialModel->getTestimonialById($id);
    }
    
    /**
     * Get featured testimonials
     * @param int $limit
     * @return array
     */
    public function getFeaturedTestimonials($limit = 3) {
        return $this->testimonialModel->getFeaturedTestimonials($limit);
    }
    
    /**
     * Get testimonials by rating
     * @param int $rating
     * @return array
     */
    public function getTestimonialsByRating($rating) {
        // Validate rating
        if (!is_numeric($rating) || $rating < 1 || $rating > 5) {
            return [];
        }
        
        return $this->testimonialModel->getTestimonialsByRating($rating);
    }
    
    /**
     * Create new testimonial
     * @param array $data
     * @return array
     */
    public function createTestimonial($data) {
        // Validate required fields
        if (empty($data['name']) || empty($data['message'])) {
            return [
                'success' => false,
                'message' => 'Name and message are required'
            ];
        }
        
        // Validate rating
        if (!isset($data['rating']) || !is_numeric($data['rating']) || $data['rating'] < 1 || $data['rating'] > 5) {
            return [
                'success' => false,
                'message' => 'Rating must be between 1 and 5'
            ];
        }
        
        // Validate message length
        if (strlen($data['message']) < 10) {
            return [
                'success' => false,
                'message' => 'Message must be at least 10 characters long'
            ];
        }
        
        if (strlen($data['message']) > 500) {
            return [
                'success' => false,
                'message' => 'Message must not exceed 500 characters'
            ];
        }
        
        // Sanitize input
        $data['name'] = htmlspecialchars(strip_tags($data['name']));
        $data['message'] = htmlspecialchars(strip_tags($data['message']));
        $data['rating'] = (int)$data['rating'];
        
        try {
            $result = $this->testimonialModel->createTestimonial($data);
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Testimonial created successfully'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Failed to create testimonial'
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Database error: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Update testimonial
     * @param int $id
     * @param array $data
     * @return array
     */
    public function updateTestimonial($id, $data) {
        // Validate ID
        if (!is_numeric($id) || $id <= 0) {
            return [
                'success' => false,
                'message' => 'Invalid testimonial ID'
            ];
        }
        
        // Check if testimonial exists
        $existingTestimonial = $this->testimonialModel->getTestimonialById($id);
        if (!$existingTestimonial) {
            return [
                'success' => false,
                'message' => 'Testimonial not found'
            ];
        }
        
        // Validate required fields
        if (empty($data['name']) || empty($data['message'])) {
            return [
                'success' => false,
                'message' => 'Name and message are required'
            ];
        }
        
        // Validate rating
        if (!isset($data['rating']) || !is_numeric($data['rating']) || $data['rating'] < 1 || $data['rating'] > 5) {
            return [
                'success' => false,
                'message' => 'Rating must be between 1 and 5'
            ];
        }
        
        // Validate message length
        if (strlen($data['message']) < 10) {
            return [
                'success' => false,
                'message' => 'Message must be at least 10 characters long'
            ];
        }
        
        if (strlen($data['message']) > 500) {
            return [
                'success' => false,
                'message' => 'Message must not exceed 500 characters'
            ];
        }
        
        // Sanitize input
        $data['name'] = htmlspecialchars(strip_tags($data['name']));
        $data['message'] = htmlspecialchars(strip_tags($data['message']));
        $data['rating'] = (int)$data['rating'];
        
        try {
            $result = $this->testimonialModel->updateTestimonial($id, $data);
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Testimonial updated successfully'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Failed to update testimonial'
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Database error: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Delete testimonial
     * @param int $id
     * @return array
     */
    public function deleteTestimonial($id) {
        // Validate ID
        if (!is_numeric($id) || $id <= 0) {
            return [
                'success' => false,
                'message' => 'Invalid testimonial ID'
            ];
        }
        
        // Check if testimonial exists
        $existingTestimonial = $this->testimonialModel->getTestimonialById($id);
        if (!$existingTestimonial) {
            return [
                'success' => false,
                'message' => 'Testimonial not found'
            ];
        }
        
        try {
            $result = $this->testimonialModel->deleteTestimonial($id);
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Testimonial deleted successfully'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Failed to delete testimonial'
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Database error: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Get average rating
     * @return float
     */
    public function getAverageRating() {
        return $this->testimonialModel->getAverageRating();
    }
    
    /**
     * Get rating distribution
     * @return array
     */
    public function getRatingDistribution() {
        return $this->testimonialModel->getRatingDistribution();
    }
    
    /**
     * Search testimonials
     * @param string $keyword
     * @return array
     */
    public function searchTestimonials($keyword) {
        // Validate keyword
        if (empty($keyword) || strlen($keyword) < 2) {
            return [];
        }
        
        $keyword = htmlspecialchars(strip_tags($keyword));
        return $this->testimonialModel->searchTestimonials($keyword);
    }
    
    /**
     * Get testimonial statistics
     * @return array
     */
    public function getTestimonialStats() {
        $totalTestimonials = $this->testimonialModel->countTestimonials();
        $averageRating = $this->testimonialModel->getAverageRating();
        $ratingDistribution = $this->testimonialModel->getRatingDistribution();
        
        return [
            'total_testimonials' => $totalTestimonials,
            'average_rating' => $averageRating,
            'rating_distribution' => $ratingDistribution
        ];
    }
}
?>