<?php
require_once 'models/ProductModel.php';

/**
 * Product Controller
 * Handles all product-related operations
 */
class ProductController {
    private $productModel;
    
    public function __construct() {
        $this->productModel = new ProductModel();
    }
    
    /**
     * Get all products
     * @return array
     */
    public function getAllProducts() {
        return $this->productModel->getAllProducts();
    }
    
    /**
     * Get product by ID
     * @param int $id
     * @return array|null
     */
    public function getProductById($id) {
        return $this->productModel->getProductById($id);
    }
    
    /**
     * Get products by category
     * @param string $category
     * @return array
     */
    public function getProductsByCategory($category) {
        return $this->productModel->getProductsByCategory($category);
    }
    
    /**
     * Get best seller products
     * @return array
     */
    public function getBestSellers() {
        return $this->productModel->getBestSellers();
    }
    
    /**
     * Create new product
     * @param array $data
     * @return array
     */
    public function createProduct($data) {
        // Validate required fields
        if (empty($data['name']) || empty($data['category']) || empty($data['image_url'])) {
            return [
                'success' => false,
                'message' => 'Name, category, and image URL are required'
            ];
        }
        
        // Sanitize input
        $data['name'] = htmlspecialchars(strip_tags($data['name']));
        $data['category'] = htmlspecialchars(strip_tags($data['category']));
        $data['description'] = htmlspecialchars(strip_tags($data['description']));
        $data['image_url'] = filter_var($data['image_url'], FILTER_SANITIZE_URL);
        $data['price'] = isset($data['price']) ? (float)$data['price'] : null;
        $data['is_best_seller'] = isset($data['is_best_seller']) ? (bool)$data['is_best_seller'] : false;
        $data['icon'] = isset($data['icon']) ? htmlspecialchars(strip_tags($data['icon'])) : '🌸';
        
        // Validate URL
        if (!filter_var($data['image_url'], FILTER_VALIDATE_URL)) {
            return [
                'success' => false,
                'message' => 'Invalid image URL'
            ];
        }
        
        try {
            $result = $this->productModel->createProduct($data);
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Product created successfully'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Failed to create product'
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Database error: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Update product
     * @param int $id
     * @param array $data
     * @return array
     */
    public function updateProduct($id, $data) {
        // Validate ID
        if (!is_numeric($id) || $id <= 0) {
            return [
                'success' => false,
                'message' => 'Invalid product ID'
            ];
        }
        
        // Check if product exists
        $existingProduct = $this->productModel->getProductById($id);
        if (!$existingProduct) {
            return [
                'success' => false,
                'message' => 'Product not found'
            ];
        }
        
        // Validate required fields
        if (empty($data['name']) || empty($data['category']) || empty($data['image_url'])) {
            return [
                'success' => false,
                'message' => 'Name, category, and image URL are required'
            ];
        }
        
        // Sanitize input
        $data['name'] = htmlspecialchars(strip_tags($data['name']));
        $data['category'] = htmlspecialchars(strip_tags($data['category']));
        $data['description'] = htmlspecialchars(strip_tags($data['description']));
        $data['image_url'] = filter_var($data['image_url'], FILTER_SANITIZE_URL);
        $data['price'] = isset($data['price']) ? (float)$data['price'] : null;
        $data['is_best_seller'] = isset($data['is_best_seller']) ? (bool)$data['is_best_seller'] : false;
        $data['icon'] = isset($data['icon']) ? htmlspecialchars(strip_tags($data['icon'])) : '🌸';
        
        // Validate URL
        if (!filter_var($data['image_url'], FILTER_VALIDATE_URL)) {
            return [
                'success' => false,
                'message' => 'Invalid image URL'
            ];
        }
        
        try {
            $result = $this->productModel->updateProduct($id, $data);
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Product updated successfully'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Failed to update product'
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Database error: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Delete product
     * @param int $id
     * @return array
     */
    public function deleteProduct($id) {
        // Validate ID
        if (!is_numeric($id) || $id <= 0) {
            return [
                'success' => false,
                'message' => 'Invalid product ID'
            ];
        }
        
        // Check if product exists
        $existingProduct = $this->productModel->getProductById($id);
        if (!$existingProduct) {
            return [
                'success' => false,
                'message' => 'Product not found'
            ];
        }
        
        try {
            $result = $this->productModel->deleteProduct($id);
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Product deleted successfully'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Failed to delete product'
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Database error: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Get product categories
     * @return array
     */
    public function getCategories() {
        return $this->productModel->getCategories();
    }
    
    /**
     * Get product statistics
     * @return array
     */
    public function getProductStats() {
        $totalProducts = $this->productModel->countProducts();
        $bestSellers = count($this->productModel->getBestSellers());
        $categories = count($this->productModel->getCategories());
        
        return [
            'total_products' => $totalProducts,
            'best_sellers' => $bestSellers,
            'categories' => $categories
        ];
    }
}
?>