<?php
require_once 'models/AgendaModel.php';

/**
 * Agenda Controller
 * Handles all agenda-related operations
 */
class AgendaController {
    private $agendaModel;
    
    public function __construct() {
        $this->agendaModel = new AgendaModel();
    }
    
    /**
     * Get all agenda items
     * @return array
     */
    public function getAllAgenda() {
        return $this->agendaModel->getAllAgenda();
    }
    
    /**
     * Get agenda by ID
     * @param int $id
     * @return array|null
     */
    public function getAgendaById($id) {
        return $this->agendaModel->getAgendaById($id);
    }
    
    /**
     * Get current month agenda
     * @return array
     */
    public function getCurrentMonthAgenda() {
        return $this->agendaModel->getCurrentMonthAgenda();
    }
    
    /**
     * Get agenda by month and year
     * @param int $month
     * @param int $year
     * @return array
     */
    public function getAgendaByMonth($month, $year) {
        // Validate month and year
        if (!is_numeric($month) || $month < 1 || $month > 12) {
            return [];
        }
        
        if (!is_numeric($year) || $year < 2020 || $year > 2030) {
            return [];
        }
        
        return $this->agendaModel->getAgendaByMonth($month, $year);
    }
    
    /**
     * Get upcoming agenda
     * @return array
     */
    public function getUpcomingAgenda() {
        return $this->agendaModel->getUpcomingAgenda();
    }
    
    /**
     * Create new agenda item
     * @param array $data
     * @return array
     */
    public function createAgenda($data) {
        // Validate required fields
        if (empty($data['date']) || empty($data['title']) || empty($data['description'])) {
            return [
                'success' => false,
                'message' => 'Date, title, and description are required'
            ];
        }
        
        // Validate date format
        if (!DateTime::createFromFormat('Y-m-d', $data['date'])) {
            return [
                'success' => false,
                'message' => 'Invalid date format. Use Y-m-d format'
            ];
        }
        
        // Extract month and year from date
        $dateObj = DateTime::createFromFormat('Y-m-d', $data['date']);
        $month = (int)$dateObj->format('n');
        $year = (int)$dateObj->format('Y');
        
        // Sanitize input
        $data['title'] = htmlspecialchars(strip_tags($data['title']));
        $data['description'] = htmlspecialchars(strip_tags($data['description']));
        $data['month'] = $month;
        $data['year'] = $year;
        
        try {
            $result = $this->agendaModel->createAgenda($data);
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Agenda item created successfully'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Failed to create agenda item'
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Database error: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Update agenda item
     * @param int $id
     * @param array $data
     * @return array
     */
    public function updateAgenda($id, $data) {
        // Validate ID
        if (!is_numeric($id) || $id <= 0) {
            return [
                'success' => false,
                'message' => 'Invalid agenda ID'
            ];
        }
        
        // Check if agenda exists
        $existingAgenda = $this->agendaModel->getAgendaById($id);
        if (!$existingAgenda) {
            return [
                'success' => false,
                'message' => 'Agenda item not found'
            ];
        }
        
        // Validate required fields
        if (empty($data['date']) || empty($data['title']) || empty($data['description'])) {
            return [
                'success' => false,
                'message' => 'Date, title, and description are required'
            ];
        }
        
        // Validate date format
        if (!DateTime::createFromFormat('Y-m-d', $data['date'])) {
            return [
                'success' => false,
                'message' => 'Invalid date format. Use Y-m-d format'
            ];
        }
        
        // Extract month and year from date
        $dateObj = DateTime::createFromFormat('Y-m-d', $data['date']);
        $month = (int)$dateObj->format('n');
        $year = (int)$dateObj->format('Y');
        
        // Sanitize input
        $data['title'] = htmlspecialchars(strip_tags($data['title']));
        $data['description'] = htmlspecialchars(strip_tags($data['description']));
        $data['month'] = $month;
        $data['year'] = $year;
        
        try {
            $result = $this->agendaModel->updateAgenda($id, $data);
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Agenda item updated successfully'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Failed to update agenda item'
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Database error: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Delete agenda item
     * @param int $id
     * @return array
     */
    public function deleteAgenda($id) {
        // Validate ID
        if (!is_numeric($id) || $id <= 0) {
            return [
                'success' => false,
                'message' => 'Invalid agenda ID'
            ];
        }
        
        // Check if agenda exists
        $existingAgenda = $this->agendaModel->getAgendaById($id);
        if (!$existingAgenda) {
            return [
                'success' => false,
                'message' => 'Agenda item not found'
            ];
        }
        
        try {
            $result = $this->agendaModel->deleteAgenda($id);
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Agenda item deleted successfully'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Failed to delete agenda item'
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Database error: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Get agenda for date range
     * @param string $startDate
     * @param string $endDate
     * @return array
     */
    public function getAgendaByDateRange($startDate, $endDate) {
        // Validate date format
        if (!DateTime::createFromFormat('Y-m-d', $startDate) || 
            !DateTime::createFromFormat('Y-m-d', $endDate)) {
            return [];
        }
        
        return $this->agendaModel->getAgendaByDateRange($startDate, $endDate);
    }
    
    /**
     * Get agenda statistics
     * @return array
     */
    public function getAgendaStats() {
        $totalAgenda = $this->agendaModel->countAgenda();
        $currentMonthAgenda = count($this->agendaModel->getCurrentMonthAgenda());
        $upcomingAgenda = count($this->agendaModel->getUpcomingAgenda());
        
        return [
            'total_agenda' => $totalAgenda,
            'current_month_agenda' => $currentMonthAgenda,
            'upcoming_agenda' => $upcomingAgenda
        ];
    }
}
?>