// Global variables
let currentProduct = {
    name: '',
    category: '',
    image: ''
};

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    initializeNavigation();
    initializeScrollEffects();
    initializeAnimations();
});

// Navigation functionality
function initializeNavigation() {
    const navLinks = document.querySelectorAll('.nav-link');
    
    // Smooth scrolling for navigation links
    navLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            const targetId = this.getAttribute('href').substring(1);
            const targetSection = document.getElementById(targetId);
            
            if (targetSection) {
                const headerHeight = document.querySelector('.header').offsetHeight;
                const targetPosition = targetSection.offsetTop - headerHeight;
                
                window.scrollTo({
                    top: targetPosition,
                    behavior: 'smooth'
                });
                
                // Update active state
                navLinks.forEach(nav => nav.classList.remove('active'));
                this.classList.add('active');
            }
        });
    });
    
    // Update active navigation on scroll
    window.addEventListener('scroll', function() {
        const sections = document.querySelectorAll('section[id]');
        const scrollPosition = window.scrollY + 100;
        
        sections.forEach(section => {
            const sectionTop = section.offsetTop;
            const sectionHeight = section.offsetHeight;
            const sectionId = section.getAttribute('id');
            
            if (scrollPosition >= sectionTop && scrollPosition < sectionTop + sectionHeight) {
                navLinks.forEach(nav => nav.classList.remove('active'));
                const activeLink = document.querySelector(`.nav-link[href="#${sectionId}"]`);
                if (activeLink) activeLink.classList.add('active');
            }
        });
    });
}

// Scroll effects
function initializeScrollEffects() {
    const header = document.querySelector('.header');
    let lastScrollTop = 0;
    
    window.addEventListener('scroll', function() {
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        
        // Header background on scroll
        if (scrollTop > 50) {
            header.style.background = 'rgba(255, 255, 255, 0.98)';
            header.style.boxShadow = '0 4px 20px rgba(0, 0, 0, 0.1)';
        } else {
            header.style.background = 'rgba(255, 255, 255, 0.95)';
            header.style.boxShadow = '0 2px 10px rgba(0, 0, 0, 0.1)';
        }
        
        lastScrollTop = scrollTop;
    });
}

// Animation on scroll
function initializeAnimations() {
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };
    
    const observer = new IntersectionObserver(function(entries) {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0)';
            }
        });
    }, observerOptions);
    
    // Observe elements for animation
    const animatedElements = document.querySelectorAll('.agenda-card, .product-card, .testimonial-card, .contact-card');
    animatedElements.forEach(el => {
        el.style.opacity = '0';
        el.style.transform = 'translateY(20px)';
        el.style.transition = 'opacity 0.6s ease, transform 0.6s ease';
        observer.observe(el);
    });
}

// Order Modal Functions
function openOrderModal(productName, productCategory, productImage) {
    currentProduct = {
        name: productName,
        category: productCategory,
        image: productImage
    };
    
    // Set modal content
    document.getElementById('modalProductName').textContent = productName;
    document.getElementById('modalProductTitle').textContent = 'Pesan ' + productName;
    document.getElementById('modalProductCategory').textContent = productCategory;
    document.getElementById('modalProductImage').src = productImage;
    document.getElementById('modalProductImage').alt = productName;
    document.getElementById('modalProductLink').href = productImage;
    document.getElementById('modalProductLink').textContent = productImage;
    
    // Show modal
    const modal = document.getElementById('orderModal');
    modal.classList.add('show');
    document.body.style.overflow = 'hidden';
    
    // Focus on first input
    setTimeout(() => {
        document.getElementById('customerName').focus();
    }, 300);
}

function closeOrderModal() {
    const modal = document.getElementById('orderModal');
    modal.classList.remove('show');
    document.body.style.overflow = 'auto';
    
    // Reset form
    document.getElementById('orderForm').reset();
}

// Image Modal Functions
function openImageModal(imageSrc) {
    document.getElementById('modalImage').src = imageSrc;
    document.getElementById('modalImageLink').href = imageSrc;
    document.getElementById('modalImageLink').textContent = imageSrc;
    
    const modal = document.getElementById('imageModal');
    modal.classList.add('show');
    document.body.style.overflow = 'hidden';
}

function closeImageModal() {
    const modal = document.getElementById('imageModal');
    modal.classList.remove('show');
    document.body.style.overflow = 'auto';
}

// Form submission
function submitOrder() {
    const name = document.getElementById('customerName').value.trim();
    const note = document.getElementById('customerNote').value.trim();
    const address = document.getElementById('customerAddress').value.trim();
    
    // Validation
    if (!name) {
        alert('Mohon masukkan nama Anda');
        document.getElementById('customerName').focus();
        return;
    }
    
    if (!address) {
        alert('Mohon masukkan alamat pengiriman');
        document.getElementById('customerAddress').focus();
        return;
    }
    
    // Create WhatsApp message
    const message = createWhatsAppMessage(name, note, address);
    
    // Open WhatsApp
    const whatsappUrl = `https://wa.me/6281223968122?text=${encodeURIComponent(message)}`;
    window.open(whatsappUrl, '_blank');
    
    // Close modal and reset form
    closeOrderModal();
}

function createWhatsAppMessage(name, note, address) {
    const noteText = note || '-';
    
    return `🌸 *PESANAN BARU - QUEEN FLORIST* 🌸

📋 *Detail Produk:*
• Nama Produk: ${currentProduct.name}
• Kategori: ${currentProduct.category}
• Lihat Foto: ${currentProduct.image}

👤 *Data Pemesan:*
• Nama: ${name}
• Catatan: ${noteText}
• Alamat Kirim: ${address}

📱 *Cara Order:*
1. Admin akan menghubungi Anda untuk konfirmasi
2. Pembayaran bisa dilakukan via Transfer/E-Wallet
3. Pengiriman sesuai alamat yang tertera

📞 *Hubungi Kami:*
• WhatsApp: +6281223968122
• Instagram: @queen_florist.kebumen

Terima kasih telah berbelanja di Queen Florist! 💐`;
}

// Event Listeners
document.addEventListener('click', function(e) {
    // Close modals when clicking outside
    if (e.target.classList.contains('modal')) {
        if (e.target.id === 'orderModal') {
            closeOrderModal();
        } else if (e.target.id === 'imageModal') {
            closeImageModal();
        }
    }
    
    // Handle product image clicks
    if (e.target.closest('#modalProductImage')) {
        openImageModal(currentProduct.image);
    }
});

// Keyboard navigation
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeOrderModal();
        closeImageModal();
    }
    
    // Form navigation with Enter key
    if (e.key === 'Enter' && e.target.tagName !== 'TEXTAREA') {
        if (e.target.id === 'customerName') {
            document.getElementById('customerNote').focus();
        } else if (e.target.id === 'customerNote') {
            document.getElementById('customerAddress').focus();
        } else if (e.target.id === 'customerAddress') {
            submitOrder();
        }
    }
});

// Add hover effects for product cards
document.addEventListener('DOMContentLoaded', function() {
    const productCards = document.querySelectorAll('.product-card');
    
    productCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-10px) scale(1.02)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0) scale(1)';
        });
    });
});

// Lazy loading for images (if needed)
function lazyLoadImages() {
    const images = document.querySelectorAll('img[data-src]');
    
    const imageObserver = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const img = entry.target;
                img.src = img.dataset.src;
                img.classList.remove('lazy');
                imageObserver.unobserve(img);
            }
        });
    });
    
    images.forEach(img => imageObserver.observe(img));
}

// Initialize lazy loading
lazyLoadImages();

// Add loading state for buttons
function addLoadingState(button, originalText) {
    button.disabled = true;
    button.innerHTML = '<span class="loading"></span> Memproses...';
    
    setTimeout(() => {
        button.disabled = false;
        button.innerHTML = originalText;
    }, 2000);
}

// Form field animations
document.addEventListener('DOMContentLoaded', function() {
    const formInputs = document.querySelectorAll('input, textarea');
    
    formInputs.forEach(input => {
        input.addEventListener('focus', function() {
            this.parentElement.classList.add('focused');
        });
        
        input.addEventListener('blur', function() {
            if (!this.value) {
                this.parentElement.classList.remove('focused');
            }
        });
    });
});

// Performance optimization - Debounce scroll events
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Apply debounce to scroll events
window.addEventListener('scroll', debounce(function() {
    // Scroll-related functions here
}, 100));

// Add touch support for mobile devices
if ('ontouchstart' in window) {
    document.body.classList.add('touch-device');
}

// Console welcome message
console.log('%c🌸 Queen Florist - Welcome! 🌸', 'color: #ec4899; font-size: 20px; font-weight: bold;');
console.log('%cBuilt with ❤️ for Queen Florist', 'color: #6b7280; font-size: 14px;');