<?php
require_once __DIR__ . '/../../config/database.php';

/**
 * Admin Product Model
 * Handles all product-related database operations for admin panel
 */
class AdminProductModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Get all products with pagination
     */
    public function getAllProducts($limit = 50, $offset = 0) {
        try {
            $stmt = $this->db->prepare("
                SELECT p.*, au.username as created_by_name, au2.username as updated_by_name 
                FROM products p
                LEFT JOIN admin_users au ON p.created_by = au.id
                LEFT JOIN admin_users au2 ON p.updated_by = au2.id
                ORDER BY p.created_at DESC 
                LIMIT :limit OFFSET :offset
            ");
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get all products error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get product by ID
     */
    public function getProductById($id) {
        try {
            $stmt = $this->db->prepare("
                SELECT p.*, au.username as created_by_name, au2.username as updated_by_name 
                FROM products p
                LEFT JOIN admin_users au ON p.created_by = au.id
                LEFT JOIN admin_users au2 ON p.updated_by = au2.id
                WHERE p.id = :id
            ");
            $stmt->execute(['id' => $id]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get product by ID error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Create new product
     */
    public function createProduct($data, $adminId = null) {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO products (name, category, description, image_url, price, is_best_seller, icon, created_by) 
                VALUES (:name, :category, :description, :image_url, :price, :is_best_seller, :icon, :created_by)
            ");
            
            $stmt->execute([
                'name' => $data['name'],
                'category' => $data['category'],
                'description' => $data['description'],
                'image_url' => $data['image_url'],
                'price' => $data['price'],
                'is_best_seller' => $data['is_best_seller'] ?? 0,
                'icon' => $data['icon'] ?? null,
                'created_by' => $adminId
            ]);
            
            return $this->db->lastInsertId();
        } catch (PDOException $e) {
            error_log("Create product error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Update product
     */
    public function updateProduct($id, $data, $adminId = null) {
        try {
            $stmt = $this->db->prepare("
                UPDATE products SET 
                    name = :name, 
                    category = :category, 
                    description = :description, 
                    image_url = :image_url, 
                    price = :price, 
                    is_best_seller = :is_best_seller, 
                    icon = :icon, 
                    updated_by = :updated_by,
                    updated_at = CURRENT_TIMESTAMP 
                WHERE id = :id
            ");
            
            $stmt->execute([
                'name' => $data['name'],
                'category' => $data['category'],
                'description' => $data['description'],
                'image_url' => $data['image_url'],
                'price' => $data['price'],
                'is_best_seller' => $data['is_best_seller'] ?? 0,
                'icon' => $data['icon'] ?? null,
                'updated_by' => $adminId,
                'id' => $id
            ]);
            
            return $stmt->rowCount() > 0;
        } catch (PDOException $e) {
            error_log("Update product error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Delete product
     */
    public function deleteProduct($id) {
        try {
            // Get product info first to delete image file
            $product = $this->getProductById($id);
            if ($product && !empty($product['image_url'])) {
                $imagePath = __DIR__ . '/../../assets/images/' . basename($product['image_url']);
                if (file_exists($imagePath)) {
                    unlink($imagePath);
                }
            }
            
            $stmt = $this->db->prepare("DELETE FROM products WHERE id = :id");
            $stmt->execute(['id' => $id]);
            return $stmt->rowCount() > 0;
        } catch (PDOException $e) {
            error_log("Delete product error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Count total products
     */
    public function countProducts() {
        try {
            $stmt = $this->db->prepare("SELECT COUNT(*) as total FROM products");
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            return $result['total'];
        } catch (PDOException $e) {
            error_log("Count products error: " . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Search products
     */
    public function searchProducts($keyword, $limit = 50, $offset = 0) {
        try {
            $stmt = $this->db->prepare("
                SELECT p.*, au.username as created_by_name, au2.username as updated_by_name 
                FROM products p
                LEFT JOIN admin_users au ON p.created_by = au.id
                LEFT JOIN admin_users au2 ON p.updated_by = au2.id
                WHERE p.name LIKE :keyword OR p.description LIKE :keyword OR p.category LIKE :keyword
                ORDER BY p.created_at DESC 
                LIMIT :limit OFFSET :offset
            ");
            
            $searchTerm = '%' . $keyword . '%';
            $stmt->bindValue(':keyword', $searchTerm);
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Search products error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get products by category
     */
    public function getProductsByCategory($category, $limit = 50, $offset = 0) {
        try {
            $stmt = $this->db->prepare("
                SELECT p.*, au.username as created_by_name, au2.username as updated_by_name 
                FROM products p
                LEFT JOIN admin_users au ON p.created_by = au.id
                LEFT JOIN admin_users au2 ON p.updated_by = au2.id
                WHERE p.category = :category
                ORDER BY p.created_at DESC 
                LIMIT :limit OFFSET :offset
            ");
            $stmt->bindValue(':category', $category);
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get products by category error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get best seller products
     */
    public function getBestSellers($limit = 10) {
        try {
            $stmt = $this->db->prepare("
                SELECT * FROM products 
                WHERE is_best_seller = 1 
                ORDER BY created_at DESC 
                LIMIT :limit
            ");
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get best sellers error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get product categories
     */
    public function getCategories() {
        try {
            $stmt = $this->db->prepare("SELECT DISTINCT category FROM products ORDER BY category");
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_COLUMN);
        } catch (PDOException $e) {
            error_log("Get categories error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Update product image only
     */
    public function updateProductImage($id, $imageUrl, $adminId = null) {
        try {
            $stmt = $this->db->prepare("
                UPDATE products SET 
                    image_url = :image_url, 
                    updated_by = :updated_by,
                    updated_at = CURRENT_TIMESTAMP 
                WHERE id = :id
            ");
            
            $stmt->execute([
                'image_url' => $imageUrl,
                'updated_by' => $adminId,
                'id' => $id
            ]);
            
            return $stmt->rowCount() > 0;
        } catch (PDOException $e) {
            error_log("Update product image error: " . $e->getMessage());
            return false;
        }
    }
}