-- =====================================================
-- QUEEN FLORIST ADMIN PANEL - COMPLETE DATABASE SCHEMA
-- =====================================================
-- Generated for production deployment
-- Compatible with MySQL 5.7+ and MariaDB 10.2+

-- Create database if not exists
CREATE DATABASE IF NOT EXISTS queen_florist CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE queen_florist;

-- =====================================================
-- DROP EXISTING TABLES (for clean re-deployment)
-- =====================================================
SET FOREIGN_KEY_CHECKS = 0;
DROP TABLE IF EXISTS admin_activity_logs;
DROP TABLE IF EXISTS admin_users;
DROP TABLE IF EXISTS testimonials;
DROP TABLE IF EXISTS agenda;
DROP TABLE IF EXISTS products;
DROP VIEW IF EXISTS admin_stats;
SET FOREIGN_KEY_CHECKS = 1;

-- =====================================================
-- ADMIN USERS TABLE
-- =====================================================
CREATE TABLE admin_users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    full_name VARCHAR(100) NOT NULL,
    role ENUM('admin', 'editor') DEFAULT 'admin',
    is_active BOOLEAN DEFAULT TRUE,
    last_login DATETIME NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_admin_users_username (username),
    INDEX idx_admin_users_email (email),
    INDEX idx_admin_users_is_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- PRODUCTS TABLE
-- =====================================================
CREATE TABLE products (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    category VARCHAR(100) NOT NULL,
    description TEXT NOT NULL,
    image_url VARCHAR(500) NULL,
    price DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    is_best_seller BOOLEAN DEFAULT FALSE,
    icon VARCHAR(50) NULL,
    created_by INT NULL,
    updated_by INT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (created_by) REFERENCES admin_users(id) ON DELETE SET NULL,
    FOREIGN KEY (updated_by) REFERENCES admin_users(id) ON DELETE SET NULL,
    
    INDEX idx_products_category (category),
    INDEX idx_products_is_best_seller (is_best_seller),
    INDEX idx_products_price (price),
    INDEX idx_products_created_at (created_at),
    INDEX idx_products_created_by (created_by)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- AGENDA TABLE
-- =====================================================
CREATE TABLE agenda (
    id INT AUTO_INCREMENT PRIMARY KEY,
    date DATE NOT NULL,
    title VARCHAR(255) NOT NULL,
    description TEXT NOT NULL,
    month INT NOT NULL,
    year INT NOT NULL,
    created_by INT NULL,
    updated_by INT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (created_by) REFERENCES admin_users(id) ON DELETE SET NULL,
    FOREIGN KEY (updated_by) REFERENCES admin_users(id) ON DELETE SET NULL,
    
    INDEX idx_agenda_date (date),
    INDEX idx_agenda_month (month),
    INDEX idx_agenda_year (year),
    INDEX idx_agenda_created_at (created_at),
    INDEX idx_agenda_created_by (created_by),
    
    UNIQUE KEY unique_agenda_date (date, title)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- TESTIMONIALS TABLE
-- =====================================================
CREATE TABLE testimonials (
    id INT AUTO_INCREMENT PRIMARY KEY,
    customer_name VARCHAR(100) NOT NULL,
    rating INT NOT NULL DEFAULT 5,
    comment TEXT NOT NULL,
    date DATE NOT NULL,
    is_visible BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_testimonials_rating (rating),
    INDEX idx_testimonials_date (date),
    INDEX idx_testimonials_is_visible (is_visible)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- ADMIN ACTIVITY LOGS TABLE
-- =====================================================
CREATE TABLE admin_activity_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    admin_id INT NOT NULL,
    action VARCHAR(100) NOT NULL,
    table_name VARCHAR(50) NULL,
    record_id INT NULL,
    description TEXT NULL,
    ip_address VARCHAR(45) NULL,
    user_agent TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (admin_id) REFERENCES admin_users(id) ON DELETE CASCADE,
    
    INDEX idx_admin_activity_logs_admin_id (admin_id),
    INDEX idx_admin_activity_logs_action (action),
    INDEX idx_admin_activity_logs_table_name (table_name),
    INDEX idx_admin_activity_logs_created_at (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- VIEWS FOR STATISTICS
-- =====================================================
CREATE VIEW admin_stats AS
SELECT 
    (SELECT COUNT(*) FROM products) as total_products,
    (SELECT COUNT(*) FROM agenda) as total_agenda,
    (SELECT COUNT(*) FROM admin_users WHERE is_active = 1) as active_admins,
    (SELECT COUNT(*) FROM admin_activity_logs WHERE DATE(created_at) = CURDATE()) as today_activities,
    (SELECT COUNT(*) FROM products WHERE is_best_seller = 1) as best_seller_products,
    (SELECT COUNT(*) FROM testimonials WHERE is_visible = 1) as visible_testimonials,
    (SELECT AVG(rating) FROM testimonials WHERE is_visible = 1) as average_rating;

-- =====================================================
-- STORED PROCEDURES
-- =====================================================

-- Get upcoming agenda items
DELIMITER //
CREATE PROCEDURE GetUpcomingAgenda(IN limit_count INT)
BEGIN
    SELECT 
        a.*,
        au.username as created_by_name
    FROM agenda a
    LEFT JOIN admin_users au ON a.created_by = au.id
    WHERE a.date >= CURDATE()
    ORDER BY a.date ASC
    LIMIT limit_count;
END //
DELIMITER ;

-- Get best seller products
DELIMITER //
CREATE PROCEDURE GetBestSellerProducts(IN limit_count INT)
BEGIN
    SELECT 
        p.*,
        au.username as created_by_name
    FROM products p
    LEFT JOIN admin_users au ON p.created_by = au.id
    WHERE p.is_best_seller = 1
    ORDER BY p.created_at DESC
    LIMIT limit_count;
END //
DELIMITER ;

-- Get recent activities
DELIMITER //
CREATE PROCEDURE GetRecentActivities(IN limit_count INT)
BEGIN
    SELECT 
        al.*,
        au.username
    FROM admin_activity_logs al
    LEFT JOIN admin_users au ON al.admin_id = au.id
    ORDER BY al.created_at DESC
    LIMIT limit_count;
END //
DELIMITER ;

-- =====================================================
-- TRIGGERS FOR AUTOMATIC LOGGING
-- =====================================================

-- Log product changes
DELIMITER //
CREATE TRIGGER products_after_insert
AFTER INSERT ON products
FOR EACH ROW
BEGIN
    INSERT INTO admin_activity_logs (admin_id, action, table_name, record_id, description)
    VALUES (NEW.created_by, 'create', 'products', NEW.id, CONCAT('Created product: ', NEW.name));
END //
DELIMITER ;

DELIMITER //
CREATE TRIGGER products_after_update
AFTER UPDATE ON products
FOR EACH ROW
BEGIN
    INSERT INTO admin_activity_logs (admin_id, action, table_name, record_id, description)
    VALUES (NEW.updated_by, 'update', 'products', NEW.id, CONCAT('Updated product: ', NEW.name));
END //
DELIMITER ;

DELIMITER //
CREATE TRIGGER products_after_delete
AFTER DELETE ON products
FOR EACH ROW
BEGIN
    INSERT INTO admin_activity_logs (admin_id, action, table_name, record_id, description)
    VALUES (NULL, 'delete', 'products', OLD.id, CONCAT('Deleted product: ', OLD.name));
END //
DELIMITER ;

-- Log agenda changes
DELIMITER //
CREATE TRIGGER agenda_after_insert
AFTER INSERT ON agenda
FOR EACH ROW
BEGIN
    INSERT INTO admin_activity_logs (admin_id, action, table_name, record_id, description)
    VALUES (NEW.created_by, 'create', 'agenda', NEW.id, CONCAT('Created agenda: ', NEW.title));
END //
DELIMITER ;

DELIMITER //
CREATE TRIGGER agenda_after_update
AFTER UPDATE ON agenda
FOR EACH ROW
BEGIN
    INSERT INTO admin_activity_logs (admin_id, action, table_name, record_id, description)
    VALUES (NEW.updated_by, 'update', 'agenda', NEW.id, CONCAT('Updated agenda: ', NEW.title));
END //
DELIMITER ;

DELIMITER //
CREATE TRIGGER agenda_after_delete
AFTER DELETE ON agenda
FOR EACH ROW
BEGIN
    INSERT INTO admin_activity_logs (admin_id, action, table_name, record_id, description)
    VALUES (NULL, 'delete', 'agenda', OLD.id, CONCAT('Deleted agenda: ', OLD.title));
END //
DELIMITER ;

-- =====================================================
-- SAMPLE DATA INSERTION
-- =====================================================

-- Insert default admin user (password: admin123)
INSERT INTO admin_users (username, password, email, full_name, role) VALUES 
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin@queenflorist.com', 'Administrator', 'admin');

-- Insert sample products
INSERT INTO products (name, category, description, image_url, price, is_best_seller, icon, created_by) VALUES 
('Bunga Papan Congratulations', 'Bunga Papan', 'Bunga papan ucapan selamat dengan desain elegan dan fresh flowers', 'assets/images/bunga-papan-1.jpg', 850000, 1, 'fa-birthday-cake', 1),
('Hand Bouquet Rose', 'Hand Bouquet', 'Buket mawar merah romantis dengan wrapping premium', 'assets/images/hand-bouquet-1.jpg', 650000, 1, 'fa-heart', 1),
('Bunga Meja Minimalis', 'Bunga Meja', 'Arrangement bunga meja minimalis untuk dekorasi ruangan', 'assets/images/bunga-meja-1.jpg', 350000, 0, 'fa-home', 1),
('Bunga Krans Duka Cita', 'Bunga Krans', 'Bunga krans untuk ucapan belasungkawa', 'assets/images/bunga-krans-1.jpg', 1200000, 0, 'fa-pray', 1),
('Box Flower Premium', 'Box Flower', 'Bunga dalam box eksklusif dengan berbagai pilihan bunga', 'assets/images/box-flower-1.jpg', 750000, 1, 'fa-gift', 1),
('Standing Flower Wedding', 'Standing Flower', 'Standing flower untuk dekorasi pernikahan', 'assets/images/standing-flower-1.jpg', 1500000, 0, 'fa-ring', 1),
('Bunga Papan Happy Wedding', 'Bunga Papan', 'Bunga papan ucapan pernikahan dengan desain mewah', 'assets/images/bunga-papan-2.jpg', 950000, 0, 'fa-heart', 1),
('Hand Bouquet Mixed', 'Hand Bouquet', 'Buket bunga campur dengan warna-warna ceria', 'assets/images/hand-bouquet-2.jpg', 550000, 0, 'fa-spa', 1),
('Bunga Meja Kantor', 'Bunga Meja', 'Arrangement bunga meja untuk dekorasi kantor', 'assets/images/bunga-meja-2.jpg', 450000, 0, 'fa-briefcase', 1),
('Standing Flower Grand Opening', 'Standing Flower', 'Standing flower untuk acara grand opening', 'assets/images/standing-flower-2.jpg', 2000000, 0, 'fa-store', 1);

-- Insert sample agenda
INSERT INTO agenda (date, title, description, month, year, created_by) VALUES 
('2025-10-15', 'Grand Opening Store', 'Peresmian pembukaan toko baru Queen Florist di Kebumen', 10, 2025, 1),
('2025-10-20', 'Flower Workshop', 'Workshop membuat arrangement bunga untuk pemula', 10, 2025, 1),
('2025-10-25', 'Flash Sale Anniversary', 'Diskon spesial ulang tahun Queen Florist', 10, 2025, 1),
('2025-11-01', 'New Collection Launch', 'Peluncuran koleksi bunga musim gugur', 11, 2025, 1);

-- Insert sample testimonials
INSERT INTO testimonials (customer_name, rating, comment, date, is_visible) VALUES 
('Siti Nurhaliza', 5, 'Pelayanan sangat memuaskan, bunga segar dan pengiriman tepat waktu. Highly recommended!', '2025-09-15', 1),
('Budi Santoso', 5, 'Kualitas bunga premium, desain elegan. Pas untuk hadiah ulang tahun istri saya.', '2025-09-20', 1),
('Maya Putri', 5, 'Queen Florist selalu menjadi pilihan utama untuk semua kebutuhan bunga saya. Terima kasih!', '2025-09-25', 1);

-- =====================================================
-- PERFORMANCE OPTIMIZATION
-- =====================================================

-- Create composite indexes for better query performance
CREATE INDEX idx_products_category_best_seller ON products(category, is_best_seller);
CREATE INDEX idx_agenda_date_title ON agenda(date, title);
CREATE INDEX idx_admin_activity_logs_admin_created ON admin_activity_logs(admin_id, created_at);

-- =====================================================
-- SECURITY SETTINGS
-- =====================================================

-- Create a read-only user for reporting (optional)
-- CREATE USER 'queen_readonly'@'localhost' IDENTIFIED BY 'secure_password';
-- GRANT SELECT ON queen_florist.* TO 'queen_readonly'@'localhost';

-- =====================================================
-- FINAL VERIFICATION
-- =====================================================

-- Verify all tables are created correctly
SELECT 
    TABLE_NAME,
    TABLE_ROWS,
    DATA_LENGTH,
    INDEX_LENGTH
FROM information_schema.TABLES 
WHERE TABLE_SCHEMA = 'queen_florist' 
ORDER BY TABLE_NAME;

-- Show summary statistics
SELECT 'Database Setup Complete' as status, NOW() as setup_time;

-- =====================================================
-- DEPLOYMENT INSTRUCTIONS
-- =====================================================
/*
1. Import this SQL file into your MySQL/MariaDB database
2. Update database configuration in config/database.php
3. Ensure the assets/images folder exists and is writable
4. Default login credentials:
   - Username: admin
   - Password: admin123
5. Change default password immediately after first login
6. Configure proper file permissions for uploads directory
7. Set up regular database backups
8. Consider enabling SSL for database connections in production

File Upload Directory Setup:
- Create: assets/images/
- Permissions: 755 (directory), 644 (files)
- Owner: web server user (www-data, apache, etc.)

Security Recommendations:
- Change default admin password
- Use strong passwords for all admin accounts
- Enable HTTPS
- Regular security updates
- Monitor admin activity logs
- Implement rate limiting for login attempts
- Use prepared statements (already implemented in PHP code)

Performance Optimization:
- Enable query cache
- Regular database maintenance
- Monitor slow queries
- Consider read replicas for high traffic
- Implement CDN for static assets
*/