<?php
require_once __DIR__ . '/../models/AdminProductModel.php';
require_once __DIR__ . '/AuthController.php';

class ProductController {
    private $productModel;
    private $adminModel;
    
    public function __construct() {
        AuthController::requireLogin();
        $this->productModel = new AdminProductModel();
        $this->adminModel = new AdminModel();
    }
    
    /**
     * Show all products
     */
    public function index() {
        $page = isset($_GET['p']) ? (int)$_GET['p'] : 1;
        $search = isset($_GET['search']) ? trim($_GET['search']) : '';
        $category = isset($_GET['category']) ? trim($_GET['category']) : '';
        $limit = 20;
        $offset = ($page - 1) * $limit;
        
        if ($search) {
            $products = $this->productModel->searchProducts($search, $limit, $offset);
            $stmt = Database::getInstance()->prepare("SELECT COUNT(*) as total FROM products WHERE name LIKE :keyword OR description LIKE :keyword OR category LIKE :keyword");
            $searchTerm = '%' . $search . '%';
            $stmt->bindValue(':keyword', $searchTerm);
            $stmt->execute();
            $total = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
        } elseif ($category) {
            $products = $this->productModel->getProductsByCategory($category, $limit, $offset);
            $stmt = Database::getInstance()->prepare("SELECT COUNT(*) as total FROM products WHERE category = :category");
            $stmt->bindValue(':category', $category);
            $stmt->execute();
            $total = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
        } else {
            $products = $this->productModel->getAllProducts($limit, $offset);
            $total = $this->productModel->countProducts();
        }
        
        $totalPages = ceil($total / $limit);
        $categories = $this->productModel->getCategories();
        
        include __DIR__ . '/../views/products/index.php';
    }
    
    /**
     * Show create product form
     */
    public function create() {
        $categories = $this->productModel->getCategories();
        include __DIR__ . '/../views/products/create.php';
    }
    
    /**
     * Store new product
     */
    public function store() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: index.php?page=products');
            exit;
        }
        
        $name = trim($_POST['name'] ?? '');
        $category = trim($_POST['category'] ?? '');
        $description = trim($_POST['description'] ?? '');
        $price = (float)($_POST['price'] ?? 0);
        $isBestSeller = isset($_POST['is_best_seller']) ? 1 : 0;
        $icon = trim($_POST['icon'] ?? '');
        
        // Validate input
        if (empty($name) || empty($category) || empty($description) || $price <= 0) {
            $_SESSION['error'] = 'Nama, kategori, deskripsi, dan harga harus diisi dengan benar';
            header('Location: index.php?page=products_create');
            exit;
        }
        
        // Handle image upload
        $imageUrl = '';
        if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
            $uploadResult = $this->handleImageUpload($_FILES['image']);
            if ($uploadResult['success']) {
                $imageUrl = $uploadResult['file_path'];
            } else {
                $_SESSION['error'] = $uploadResult['message'];
                header('Location: index.php?page=products_create');
                exit;
            }
        } else {
            $_SESSION['error'] = 'Gambar produk harus diupload';
            header('Location: index.php?page=products_create');
            exit;
        }
        
        $data = [
            'name' => $name,
            'category' => $category,
            'description' => $description,
            'image_url' => $imageUrl,
            'price' => $price,
            'is_best_seller' => $isBestSeller,
            'icon' => $icon
        ];
        
        $productId = $this->productModel->createProduct($data, $_SESSION['admin_id']);
        
        if ($productId) {
            $this->adminModel->logActivity($_SESSION['admin_id'], 'create', 'products', $productId, "Created product: $name");
            $_SESSION['success'] = 'Produk berhasil ditambahkan';
            header('Location: index.php?page=products');
        } else {
            $_SESSION['error'] = 'Gagal menambahkan produk';
            header('Location: index.php?page=products_create');
        }
        exit;
    }
    
    /**
     * Show edit product form
     */
    public function edit($id) {
        $product = $this->productModel->getProductById($id);
        $categories = $this->productModel->getCategories();
        
        if (!$product) {
            $_SESSION['error'] = 'Produk tidak ditemukan';
            header('Location: index.php?page=products');
            exit;
        }
        
        include __DIR__ . '/../views/products/edit.php';
    }
    
    /**
     * Update product
     */
    public function update($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: index.php?page=products');
            exit;
        }
        
        $name = trim($_POST['name'] ?? '');
        $category = trim($_POST['category'] ?? '');
        $description = trim($_POST['description'] ?? '');
        $price = (float)($_POST['price'] ?? 0);
        $isBestSeller = isset($_POST['is_best_seller']) ? 1 : 0;
        $icon = trim($_POST['icon'] ?? '');
        
        // Validate input
        if (empty($name) || empty($category) || empty($description) || $price <= 0) {
            $_SESSION['error'] = 'Nama, kategori, deskripsi, dan harga harus diisi dengan benar';
            header('Location: index.php?page=products_edit&id=' . $id);
            exit;
        }
        
        // Check if product exists
        $existingProduct = $this->productModel->getProductById($id);
        if (!$existingProduct) {
            $_SESSION['error'] = 'Produk tidak ditemukan';
            header('Location: index.php?page=products');
            exit;
        }
        
        // Handle image upload
        $imageUrl = $existingProduct['image_url']; // Keep existing image by default
        if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
            $uploadResult = $this->handleImageUpload($_FILES['image']);
            if ($uploadResult['success']) {
                // Delete old image if exists
                if (!empty($existingProduct['image_url'])) {
                    $oldImagePath = __DIR__ . '/../../assets/images/' . basename($existingProduct['image_url']);
                    if (file_exists($oldImagePath)) {
                        unlink($oldImagePath);
                    }
                }
                $imageUrl = $uploadResult['file_path'];
            } else {
                $_SESSION['error'] = $uploadResult['message'];
                header('Location: index.php?page=products_edit&id=' . $id);
                exit;
            }
        }
        
        $data = [
            'name' => $name,
            'category' => $category,
            'description' => $description,
            'image_url' => $imageUrl,
            'price' => $price,
            'is_best_seller' => $isBestSeller,
            'icon' => $icon
        ];
        
        $success = $this->productModel->updateProduct($id, $data, $_SESSION['admin_id']);
        
        if ($success) {
            $this->adminModel->logActivity($_SESSION['admin_id'], 'update', 'products', $id, "Updated product: $name");
            $_SESSION['success'] = 'Produk berhasil diperbarui';
        } else {
            $_SESSION['error'] = 'Gagal memperbarui produk';
        }
        
        header('Location: index.php?page=products');
        exit;
    }
    
    /**
     * Delete product
     */
    public function delete($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: index.php?page=products');
            exit;
        }
        
        // Check if product exists
        $product = $this->productModel->getProductById($id);
        if (!$product) {
            $_SESSION['error'] = 'Produk tidak ditemukan';
            header('Location: index.php?page=products');
            exit;
        }
        
        $success = $this->productModel->deleteProduct($id);
        
        if ($success) {
            $this->adminModel->logActivity($_SESSION['admin_id'], 'delete', 'products', $id, "Deleted product: {$product['name']}");
            $_SESSION['success'] = 'Produk berhasil dihapus';
        } else {
            $_SESSION['error'] = 'Gagal menghapus produk';
        }
        
        header('Location: index.php?page=products');
        exit;
    }
    
    /**
     * View product details
     */
    public function view($id) {
        $product = $this->productModel->getProductById($id);
        
        if (!$product) {
            $_SESSION['error'] = 'Produk tidak ditemukan';
            header('Location: index.php?page=products');
            exit;
        }
        
        include __DIR__ . '/../views/products/view.php';
    }
    
    /**
     * Bulk delete products
     */
    public function bulkDelete() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: index.php?page=products');
            exit;
        }
        
        $ids = $_POST['ids'] ?? [];
        
        if (empty($ids)) {
            $_SESSION['error'] = 'Pilih minimal satu produk untuk dihapus';
            header('Location: index.php?page=products');
            exit;
        }
        
        $deletedCount = 0;
        foreach ($ids as $id) {
            $product = $this->productModel->getProductById($id);
            if ($product && $this->productModel->deleteProduct($id)) {
                $this->adminModel->logActivity($_SESSION['admin_id'], 'delete', 'products', $id, "Deleted product: {$product['name']}");
                $deletedCount++;
            }
        }
        
        if ($deletedCount > 0) {
            $_SESSION['success'] = "$deletedCount produk berhasil dihapus";
        } else {
            $_SESSION['error'] = 'Tidak ada produk yang berhasil dihapus';
        }
        
        header('Location: index.php?page=products');
        exit;
    }
    
    /**
     * Handle image upload
     */
    private function handleImageUpload($file) {
        $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
        $maxSize = 5 * 1024 * 1024; // 5MB
        $uploadDir = __DIR__ . '/../../assets/images/';
        
        // Check if upload directory exists, create if not
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        
        // Validate file
        if (!in_array($file['type'], $allowedTypes)) {
            return ['success' => false, 'message' => 'Tipe file tidak diizinkan. Hanya JPG, PNG, GIF, dan WebP.'];
        }
        
        if ($file['size'] > $maxSize) {
            return ['success' => false, 'message' => 'Ukuran file terlalu besar. Maksimal 5MB.'];
        }
        
        // Generate unique filename
        $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $filename = 'product_' . time() . '_' . uniqid() . '.' . $extension;
        $filepath = $uploadDir . $filename;
        
        // Move uploaded file
        if (move_uploaded_file($file['tmp_name'], $filepath)) {
            // Return relative path for database storage
            return ['success' => true, 'file_path' => 'assets/images/' . $filename];
        } else {
            return ['success' => false, 'message' => 'Gagal mengupload file.'];
        }
    }
    
    /**
     * Toggle best seller status
     */
    public function toggleBestSeller($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: index.php?page=products');
            exit;
        }
        
        $product = $this->productModel->getProductById($id);
        if (!$product) {
            $_SESSION['error'] = 'Produk tidak ditemukan';
            header('Location: index.php?page=products');
            exit;
        }
        
        $newStatus = $product['is_best_seller'] ? 0 : 1;
        $data = [
            'name' => $product['name'],
            'category' => $product['category'],
            'description' => $product['description'],
            'image_url' => $product['image_url'],
            'price' => $product['price'],
            'is_best_seller' => $newStatus,
            'icon' => $product['icon']
        ];
        
        $success = $this->productModel->updateProduct($id, $data, $_SESSION['admin_id']);
        
        if ($success) {
            $statusText = $newStatus ? 'dijadikan best seller' : 'dihapus dari best seller';
            $this->adminModel->logActivity($_SESSION['admin_id'], 'toggle_best_seller', 'products', $id, "Product {$product['name']} $statusText");
            $_SESSION['success'] = 'Status best seller berhasil diubah';
        } else {
            $_SESSION['error'] = 'Gagal mengubah status best seller';
        }
        
        header('Location: index.php?page=products');
        exit;
    }
}