<?php
require_once __DIR__ . '/../models/AdminModel.php';
require_once __DIR__ . '/../models/AdminAgendaModel.php';
require_once __DIR__ . '/../models/AdminProductModel.php';
require_once __DIR__ . '/AuthController.php';

class DashboardController {
    private $adminModel;
    private $agendaModel;
    private $productModel;
    
    public function __construct() {
        AuthController::requireLogin();
        $this->adminModel = new AdminModel();
        $this->agendaModel = new AdminAgendaModel();
        $this->productModel = new AdminProductModel();
    }
    
    /**
     * Show dashboard
     */
    public function showDashboard() {
        $stats = $this->adminModel->getStats();
        $recentActivities = $this->adminModel->getActivityLogs(10);
        $upcomingAgenda = $this->agendaModel->getUpcomingAgenda(5);
        $recentProducts = $this->productModel->getAllProducts(5);
        
        include __DIR__ . '/../views/dashboard.php';
    }
    
    /**
     * Show activity logs
     */
    public function showActivityLogs() {
        $page = isset($_GET['p']) ? (int)$_GET['p'] : 1;
        $limit = 20;
        $offset = ($page - 1) * $limit;
        
        $logs = $this->adminModel->getActivityLogs($limit, $offset);
        
        // Get total count for pagination
        $stmt = Database::getInstance()->prepare("SELECT COUNT(*) as total FROM admin_activity_logs");
        $stmt->execute();
        $total = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
        $totalPages = ceil($total / $limit);
        
        include __DIR__ . '/../views/activity_logs.php';
    }
    
    /**
     * Show profile
     */
    public function showProfile() {
        $admin = $this->adminModel->getAdminById($_SESSION['admin_id']);
        include __DIR__ . '/../views/profile.php';
    }
    
    /**
     * Update profile
     */
    public function updateProfile() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: index.php?page=profile');
            exit;
        }
        
        $fullName = trim($_POST['full_name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        
        // Validate input
        if (empty($fullName) || empty($email)) {
            $_SESSION['error'] = 'Nama lengkap dan email harus diisi';
            header('Location: index.php?page=profile');
            exit;
        }
        
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $_SESSION['error'] = 'Format email tidak valid';
            header('Location: index.php?page=profile');
            exit;
        }
        
        try {
            $stmt = Database::getInstance()->prepare("
                UPDATE admin_users 
                SET full_name = :full_name, email = :email, updated_at = NOW() 
                WHERE id = :id
            ");
            
            $stmt->execute([
                'full_name' => $fullName,
                'email' => $email,
                'id' => $_SESSION['admin_id']
            ]);
            
            // Update session
            $_SESSION['admin_name'] = $fullName;
            
            $this->adminModel->logActivity($_SESSION['admin_id'], 'update_profile', null, null, 'Profile updated');
            $_SESSION['success'] = 'Profil berhasil diperbarui';
            
        } catch (PDOException $e) {
            error_log("Update profile error: " . $e->getMessage());
            $_SESSION['error'] = 'Gagal memperbarui profil';
        }
        
        header('Location: index.php?page=profile');
        exit;
    }
    
    /**
     * Show admin management (admin only)
     */
    public function showAdminManagement() {
        // Only admin can access this
        if ($_SESSION['admin_role'] !== 'admin') {
            $_SESSION['error'] = 'Anda tidak memiliki akses ke halaman ini';
            header('Location: index.php?page=dashboard');
            exit;
        }
        
        $admins = $this->adminModel->getAllAdmins();
        include __DIR__ . '/../views/admin_management.php';
    }
    
    /**
     * Create new admin (admin only)
     */
    public function createAdmin() {
        if ($_SESSION['admin_role'] !== 'admin') {
            $_SESSION['error'] = 'Anda tidak memiliki akses';
            header('Location: index.php?page=dashboard');
            exit;
        }
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: index.php?page=admin_management');
            exit;
        }
        
        $username = trim($_POST['username'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $fullName = trim($_POST['full_name'] ?? '');
        $password = $_POST['password'] ?? '';
        $role = $_POST['role'] ?? 'editor';
        
        // Validate input
        if (empty($username) || empty($email) || empty($fullName) || empty($password)) {
            $_SESSION['error'] = 'Semua field harus diisi';
            header('Location: index.php?page=admin_management');
            exit;
        }
        
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $_SESSION['error'] = 'Format email tidak valid';
            header('Location: index.php?page=admin_management');
            exit;
        }
        
        if (strlen($password) < 6) {
            $_SESSION['error'] = 'Password minimal 6 karakter';
            header('Location: index.php?page=admin_management');
            exit;
        }
        
        $adminId = $this->adminModel->createAdmin([
            'username' => $username,
            'email' => $email,
            'full_name' => $fullName,
            'password' => $password,
            'role' => $role
        ]);
        
        if ($adminId) {
            $this->adminModel->logActivity($_SESSION['admin_id'], 'create_admin', 'admin_users', $adminId, "Created admin user: $username");
            $_SESSION['success'] = 'Admin berhasil dibuat';
        } else {
            $_SESSION['error'] = 'Gagal membuat admin. Username atau email mungkin sudah digunakan';
        }
        
        header('Location: index.php?page=admin_management');
        exit;
    }
    
    /**
     * Toggle admin status (admin only)
     */
    public function toggleAdminStatus($id) {
        if ($_SESSION['admin_role'] !== 'admin') {
            $_SESSION['error'] = 'Anda tidak memiliki akses';
            header('Location: index.php?page=dashboard');
            exit;
        }
        
        if ($id == $_SESSION['admin_id']) {
            $_SESSION['error'] = 'Tidak dapat mengubah status sendiri';
            header('Location: index.php?page=admin_management');
            exit;
        }
        
        try {
            $stmt = Database::getInstance()->prepare("
                UPDATE admin_users 
                SET is_active = NOT is_active 
                WHERE id = :id
            ");
            $stmt->execute(['id' => $id]);
            
            $this->adminModel->logActivity($_SESSION['admin_id'], 'toggle_admin_status', 'admin_users', $id, "Toggled admin status for ID: $id");
            $_SESSION['success'] = 'Status admin berhasil diubah';
            
        } catch (PDOException $e) {
            error_log("Toggle admin status error: " . $e->getMessage());
            $_SESSION['error'] = 'Gagal mengubah status admin';
        }
        
        header('Location: index.php?page=admin_management');
        exit;
    }
}