<?php
session_start();
require_once __DIR__ . '/../models/AdminModel.php';

class AuthController {
    private $adminModel;
    
    public function __construct() {
        $this->adminModel = new AdminModel();
    }
    
    /**
     * Show login page
     */
    public function showLogin() {
        // If already logged in, redirect to dashboard
        if (isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true) {
            header('Location: index.php?page=dashboard');
            exit;
        }
        
        include __DIR__ . '/../views/login.php';
    }
    
    /**
     * Process login
     */
    public function login() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: index.php?page=login');
            exit;
        }
        
        $username = trim($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';
        
        // Validate input
        if (empty($username) || empty($password)) {
            $_SESSION['error'] = 'Username dan password harus diisi';
            header('Location: index.php?page=login');
            exit;
        }
        
        // Attempt login
        $user = $this->adminModel->login($username, $password);
        
        if ($user) {
            // Set session
            $_SESSION['admin_logged_in'] = true;
            $_SESSION['admin_id'] = $user['id'];
            $_SESSION['admin_username'] = $user['username'];
            $_SESSION['admin_name'] = $user['full_name'];
            $_SESSION['admin_role'] = $user['role'];
            
            // Log activity
            $this->adminModel->logActivity($user['id'], 'login', null, null, 'Admin login successful');
            
            // Redirect to dashboard
            header('Location: index.php?page=dashboard');
            exit;
        } else {
            $_SESSION['error'] = 'Username atau password salah';
            header('Location: index.php?page=login');
            exit;
        }
    }
    
    /**
     * Logout
     */
    public function logout() {
        if (isset($_SESSION['admin_id'])) {
            $this->adminModel->logActivity($_SESSION['admin_id'], 'logout', null, null, 'Admin logout');
        }
        
        // Destroy session
        session_destroy();
        
        // Redirect to login
        header('Location: index.php?page=login');
        exit;
    }
    
    /**
     * Check if user is logged in
     */
    public static function isLoggedIn() {
        return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
    }
    
    /**
     * Require login
     */
    public static function requireLogin() {
        if (!self::isLoggedIn()) {
            header('Location: index.php?page=login');
            exit;
        }
    }
    
    /**
     * Get current admin user
     */
    public static function getCurrentUser() {
        if (self::isLoggedIn()) {
            return [
                'id' => $_SESSION['admin_id'],
                'username' => $_SESSION['admin_username'],
                'name' => $_SESSION['admin_name'],
                'role' => $_SESSION['admin_role']
            ];
        }
        return null;
    }
    
    /**
     * Show change password form
     */
    public function showChangePassword() {
        self::requireLogin();
        include __DIR__ . '/../views/change_password.php';
    }
    
    /**
     * Process change password
     */
    public function changePassword() {
        self::requireLogin();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: index.php?page=change_password');
            exit;
        }
        
        $currentPassword = $_POST['current_password'] ?? '';
        $newPassword = $_POST['new_password'] ?? '';
        $confirmPassword = $_POST['confirm_password'] ?? '';
        
        // Validate input
        if (empty($currentPassword) || empty($newPassword) || empty($confirmPassword)) {
            $_SESSION['error'] = 'Semua field harus diisi';
            header('Location: index.php?page=change_password');
            exit;
        }
        
        if ($newPassword !== $confirmPassword) {
            $_SESSION['error'] = 'Password baru dan konfirmasi tidak cocok';
            header('Location: index.php?page=change_password');
            exit;
        }
        
        if (strlen($newPassword) < 6) {
            $_SESSION['error'] = 'Password baru minimal 6 karakter';
            header('Location: index.php?page=change_password');
            exit;
        }
        
        // Change password
        $success = $this->adminModel->changePassword($_SESSION['admin_id'], $currentPassword, $newPassword);
        
        if ($success) {
            $this->adminModel->logActivity($_SESSION['admin_id'], 'change_password', null, null, 'Password changed successfully');
            $_SESSION['success'] = 'Password berhasil diubah';
        } else {
            $_SESSION['error'] = 'Password saat ini salah';
        }
        
        header('Location: index.php?page=change_password');
        exit;
    }
}