<?php
require_once __DIR__ . '/../models/AdminAgendaModel.php';
require_once __DIR__ . '/AuthController.php';

class AgendaController {
    private $agendaModel;
    private $adminModel;
    
    public function __construct() {
        AuthController::requireLogin();
        $this->agendaModel = new AdminAgendaModel();
        $this->adminModel = new AdminModel();
    }
    
    /**
     * Show all agenda items
     */
    public function index() {
        $page = isset($_GET['p']) ? (int)$_GET['p'] : 1;
        $search = isset($_GET['search']) ? trim($_GET['search']) : '';
        $limit = 20;
        $offset = ($page - 1) * $limit;
        
        if ($search) {
            $agenda = $this->agendaModel->searchAgenda($search, $limit, $offset);
            $stmt = Database::getInstance()->prepare("SELECT COUNT(*) as total FROM agenda WHERE title LIKE :keyword OR description LIKE :keyword OR date LIKE :keyword");
            $searchTerm = '%' . $search . '%';
            $stmt->bindValue(':keyword', $searchTerm);
            $stmt->execute();
            $total = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
        } else {
            $agenda = $this->agendaModel->getAllAgenda($limit, $offset);
            $total = $this->agendaModel->countAgenda();
        }
        
        $totalPages = ceil($total / $limit);
        
        include __DIR__ . '/../views/agenda/index.php';
    }
    
    /**
     * Show create agenda form
     */
    public function create() {
        include __DIR__ . '/../views/agenda/create.php';
    }
    
    /**
     * Store new agenda
     */
    public function store() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: index.php?page=agenda');
            exit;
        }
        
        $title = trim($_POST['title'] ?? '');
        $description = trim($_POST['description'] ?? '');
        $date = $_POST['date'] ?? '';
        
        // Validate input
        if (empty($title) || empty($description) || empty($date)) {
            $_SESSION['error'] = 'Judul, deskripsi, dan tanggal harus diisi';
            header('Location: index.php?page=agenda_create');
            exit;
        }
        
        // Validate date format
        if (!DateTime::createFromFormat('Y-m-d', $date)) {
            $_SESSION['error'] = 'Format tanggal tidak valid';
            header('Location: index.php?page=agenda_create');
            exit;
        }
        
        $data = [
            'title' => $title,
            'description' => $description,
            'date' => $date
        ];
        
        $agendaId = $this->agendaModel->createAgenda($data, $_SESSION['admin_id']);
        
        if ($agendaId) {
            $this->adminModel->logActivity($_SESSION['admin_id'], 'create', 'agenda', $agendaId, "Created agenda: $title");
            $_SESSION['success'] = 'Agenda berhasil ditambahkan';
            header('Location: index.php?page=agenda');
        } else {
            $_SESSION['error'] = 'Gagal menambahkan agenda';
            header('Location: index.php?page=agenda_create');
        }
        exit;
    }
    
    /**
     * Show edit agenda form
     */
    public function edit($id) {
        $agenda = $this->agendaModel->getAgendaById($id);
        
        if (!$agenda) {
            $_SESSION['error'] = 'Agenda tidak ditemukan';
            header('Location: index.php?page=agenda');
            exit;
        }
        
        include __DIR__ . '/../views/agenda/edit.php';
    }
    
    /**
     * Update agenda
     */
    public function update($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: index.php?page=agenda');
            exit;
        }
        
        $title = trim($_POST['title'] ?? '');
        $description = trim($_POST['description'] ?? '');
        $date = $_POST['date'] ?? '';
        
        // Validate input
        if (empty($title) || empty($description) || empty($date)) {
            $_SESSION['error'] = 'Judul, deskripsi, dan tanggal harus diisi';
            header('Location: index.php?page=agenda_edit&id=' . $id);
            exit;
        }
        
        // Validate date format
        if (!DateTime::createFromFormat('Y-m-d', $date)) {
            $_SESSION['error'] = 'Format tanggal tidak valid';
            header('Location: index.php?page=agenda_edit&id=' . $id);
            exit;
        }
        
        // Check if agenda exists
        $existingAgenda = $this->agendaModel->getAgendaById($id);
        if (!$existingAgenda) {
            $_SESSION['error'] = 'Agenda tidak ditemukan';
            header('Location: index.php?page=agenda');
            exit;
        }
        
        $data = [
            'title' => $title,
            'description' => $description,
            'date' => $date
        ];
        
        $success = $this->agendaModel->updateAgenda($id, $data, $_SESSION['admin_id']);
        
        if ($success) {
            $this->adminModel->logActivity($_SESSION['admin_id'], 'update', 'agenda', $id, "Updated agenda: $title");
            $_SESSION['success'] = 'Agenda berhasil diperbarui';
        } else {
            $_SESSION['error'] = 'Gagal memperbarui agenda';
        }
        
        header('Location: index.php?page=agenda');
        exit;
    }
    
    /**
     * Delete agenda
     */
    public function delete($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: index.php?page=agenda');
            exit;
        }
        
        // Check if agenda exists
        $agenda = $this->agendaModel->getAgendaById($id);
        if (!$agenda) {
            $_SESSION['error'] = 'Agenda tidak ditemukan';
            header('Location: index.php?page=agenda');
            exit;
        }
        
        $success = $this->agendaModel->deleteAgenda($id);
        
        if ($success) {
            $this->adminModel->logActivity($_SESSION['admin_id'], 'delete', 'agenda', $id, "Deleted agenda: {$agenda['title']}");
            $_SESSION['success'] = 'Agenda berhasil dihapus';
        } else {
            $_SESSION['error'] = 'Gagal menghapus agenda';
        }
        
        header('Location: index.php?page=agenda');
        exit;
    }
    
    /**
     * View agenda details
     */
    public function view($id) {
        $agenda = $this->agendaModel->getAgendaById($id);
        
        if (!$agenda) {
            $_SESSION['error'] = 'Agenda tidak ditemukan';
            header('Location: index.php?page=agenda');
            exit;
        }
        
        include __DIR__ . '/../views/agenda/view.php';
    }
    
    /**
     * Bulk delete agenda items
     */
    public function bulkDelete() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: index.php?page=agenda');
            exit;
        }
        
        $ids = $_POST['ids'] ?? [];
        
        if (empty($ids)) {
            $_SESSION['error'] = 'Pilih minimal satu agenda untuk dihapus';
            header('Location: index.php?page=agenda');
            exit;
        }
        
        $deletedCount = 0;
        foreach ($ids as $id) {
            $agenda = $this->agendaModel->getAgendaById($id);
            if ($agenda && $this->agendaModel->deleteAgenda($id)) {
                $this->adminModel->logActivity($_SESSION['admin_id'], 'delete', 'agenda', $id, "Deleted agenda: {$agenda['title']}");
                $deletedCount++;
            }
        }
        
        if ($deletedCount > 0) {
            $_SESSION['success'] = "$deletedCount agenda berhasil dihapus";
        } else {
            $_SESSION['error'] = 'Tidak ada agenda yang berhasil dihapus';
        }
        
        header('Location: index.php?page=agenda');
        exit;
    }
}